import pygame, math
pygame.init()

# ----------------------------- Window & timing -----------------------------
W, H = 800, 600
screen = pygame.display.set_mode((W, H))
pygame.display.set_caption("Stealth Pickup")
clock = pygame.time.Clock()
font = pygame.font.SysFont(None, 28)

# ----------------------------- Helpers ------------------------------------
def get_distance(a, b):
    ax, ay = a.center
    bx, by = b.center
    return math.hypot(ax - bx, ay - by)

def move_towards(rect, target, speed, dt):
    dx = target[0] - rect.centerx
    dy = target[1] - rect.centery
    dist = math.hypot(dx, dy)
    if dist > 0:
        dx /= dist; dy /= dist
        rect.x += int(dx * speed * dt)
        rect.y += int(dy * speed * dt)

def clear_view(from_rect, to_rect, walls):
    ax, ay = from_rect.center
    bx, by = to_rect.center
    dist = math.hypot(bx - ax, by - ay)
    if dist == 0:
        return True
    steps = max(1, int(dist // 4))
    for i in range(steps + 1):
        t = i / steps
        x = ax + (bx - ax) * t
        y = ay + (by - ay) * t
        for w in walls:
            if w.collidepoint(x, y):
                return False
    return True

def clamp_rect(rect, w, h):
    if rect.left < 0: rect.left = 0
    if rect.top < 0: rect.top = 0
    if rect.right > w: rect.right = w
    if rect.bottom > h: rect.bottom = h

# ----------------------------- Sprites (surfaces) --------------------------
player_img  = pygame.Surface((28, 28), pygame.SRCALPHA); player_img.fill((50, 200, 255))
enemy_img   = pygame.Surface((28, 28), pygame.SRCALPHA); enemy_img.fill((255, 80, 80))
treasure_img= pygame.Surface((24, 24), pygame.SRCALPHA); treasure_img.fill((255, 220, 0))

# Walls (store rects in `walls`, keep surfaces in parallel list for drawing)
walls = []
def make_wall(x, y, w, h):
    s = pygame.Surface((w, h), pygame.SRCALPHA); s.fill((120, 120, 120))
    r = s.get_rect(topleft=(x, y))
    walls.append(r)
    return s

# --- Layout to match the screenshot ---
# Thickness ~30px; corridors long; thin vertical caps at TL and BR.
wall_surfaces = [
    make_wall(120,   0, 30, 150),   # top-left thin vertical
    make_wall(120, 120, 650, 30),   # top corridor (long)
    make_wall(340, 320, 250, 30),   # central cover
    make_wall(120, 450, 650, 30),   # bottom corridor (long)
    make_wall(770, 450, 30, 150),   # bottom-right thin vertical
]

# ----------------------------- Entities ------------------------------------
player_rect   = player_img.get_rect(center=(70, 560))    # bottom-left
treasure_rect = treasure_img.get_rect(center=(740, 80))  # top-right

class Enemy:
    def __init__(self, image, start_pos, points):
        self.image = image
        self.rect = image.get_rect(center=start_pos)
        self.points = points[:]
        self.point_index = 0
        self.state = "PATROL"
        self.view_range = 260
        self.speed_patrol = 90
        self.speed_chase  = 150
        self.cooldown_time = 1200
        self.lost_time = 0
        self.last_seen = self.rect.center
        self.sees = False

    def update(self, dt, player_rect):
        dist = get_distance(self.rect, player_rect)
        self.sees = (dist <= self.view_range) and clear_view(self.rect, player_rect, walls)

        if self.state == "PATROL":
            target = self.points[self.point_index]
            move_towards(self.rect, target, self.speed_patrol, dt)
            if pygame.Vector2(self.rect.center).distance_to(target) < 4:
                self.point_index = (self.point_index + 1) % len(self.points)
            if self.sees:
                self.state = "CHASE"

        elif self.state == "CHASE":
            self.last_seen = player_rect.center
            move_towards(self.rect, self.last_seen, self.speed_chase, dt)
            if not self.sees:
                self.state = "COOLDOWN"
                self.lost_time = pygame.time.get_ticks()

        elif self.state == "COOLDOWN":
            move_towards(self.rect, self.last_seen, self.speed_patrol, dt)
            if self.sees:
                self.state = "CHASE"
            elif pygame.time.get_ticks() - self.lost_time >= self.cooldown_time:
                self.state = "PATROL"

# Enemy mid-left (just above bottom corridor), patrolling a big loop
enemy = Enemy(enemy_img, (160, 390), [(700, 520), (700, 300), (100, 300), (100, 520)])

# ----------------------------- Player movement -----------------------------
PLAYER_SPEED = 200
def move_player_with_blocking(rect, dx, dy):
    rect.x += dx
    for w in walls:
        if rect.colliderect(w):
            if dx > 0: rect.right = w.left
            elif dx < 0: rect.left = w.right
    rect.y += dy
    for w in walls:
        if rect.colliderect(w):
            if dy > 0: rect.bottom = w.top
            elif dy < 0: rect.top = w.bottom
    clamp_rect(rect, W, H)

# ----------------------------- Game loop -----------------------------------
state = "PLAY"  # PLAY, WIN, LOSE

def reset():
    global state, player_rect, enemy
    state = "PLAY"
    player_rect.center = (70, 560)
    enemy = Enemy(enemy_img, (160, 390), [(700, 520), (700, 300), (100, 300), (100, 520)])

running = True
while running:
    dt = clock.tick(60) / 1000.0

    for event in pygame.event.get():
        if event.type == pygame.QUIT:
            running = False

    keys = pygame.key.get_pressed()

    if state == "PLAY":
        dx = dy = 0
        if keys[pygame.K_LEFT]:  dx -= int(PLAYER_SPEED * dt)
        if keys[pygame.K_RIGHT]: dx += int(PLAYER_SPEED * dt)
        if keys[pygame.K_UP]:    dy -= int(PLAYER_SPEED * dt)
        if keys[pygame.K_DOWN]:  dy += int(PLAYER_SPEED * dt)
        move_player_with_blocking(player_rect, dx, dy)

        enemy.update(dt, player_rect)

        if player_rect.colliderect(treasure_rect):
            state = "WIN"
        if enemy.rect.colliderect(player_rect):
            state = "LOSE"

    if keys[pygame.K_r]:
        reset()

    # Draw
    screen.fill((18, 18, 22))
    for s, r in zip(wall_surfaces, walls):
        screen.blit(s, r)
    screen.blit(treasure_img, treasure_rect)
    screen.blit(enemy.image, enemy.rect)
    screen.blit(player_img, player_rect)

    # UI text
    if state == "PLAY":
        txt = font.render("Reach the treasure. Hide behind walls. Arrows to move. R to reset.", True, (240, 240, 240))
    elif state == "WIN":
        txt = font.render("You win! Press R to play again.", True, (50, 220, 80))
    else:
        txt = font.render("Caught! Press R to retry.", True, (240, 80, 80))
    screen.blit(txt, (20, H - 36))

    pygame.display.flip()

pygame.quit()
